;*******  zx_incl.asm  *******
;
; Roelh  251207
;
; include this file to run ZX spectrum programs on Isetta (using CP/M)
;


 org #100 ; start of CP/M program

 
zx_stack equ #0100 ; stack during the CP/M startup phase

kbhandler equ #3800
kbhandler_msb equ #38

 
vga_frame equ #0700  ; in bank 1 ! 
zx_bitmap equ #4000  ; bank 0.  end-of-line flags in bank 1
zx_attribs equ #5800 ; bank 0
zx_color_trans equ #0800 ; 15-11-2024 color translation table in bank 0 (FG) and bank 2 (BG). 256 values each.

; video processor commands;

v_prepare equ #24
v_end     equ #25
v_blank   equ #26
v_wait    equ #27
v_act_vsync   equ #2A
v_deact_vsync equ #2B
v_repeat      equ #2C
; commands for video modes;
v_hires       equ #2D ; 13-5-2024
v_spectrum    equ #0D ; 7-6-2024

; i/o ports to read/write from/to different memory bank;

port_io_bank0 equ #F0 ; 15-11-2024
port_io_bank1 equ #F1
port_io_bank2 equ #F2
port_io_bank3 equ #F3

port_in_scancode equ #31 ; input scancode from PS/2 keyboard

port_io_reg_vga_ptl equ #11  ; use for a timer 

port_out_sound_mode equ #43 ; 1-9-2025

port_out_system equ #18 ; 22-12-2025

col_red   equ #04
col_green equ #02
col_blue  equ #01
; some bit definitions
bit0 equ 1
bit1 equ 2
bit2 equ 4
bit3 equ 8
bit4 equ #10
bit5 equ #20
bit6 equ #40
bit7 equ #80

 
;*** program start ***
 
 ld sp,zx_stack
 di
 call zx_set_interrupt 
 call zx_video_init
 call zx_kb_init
 jp zx_main
 

;***********  Init ZX SPECTRUM  KEYBOARD  **********

zx_kb_init:
 ld a,kbhandler_msb   ; MSB of keyboard function
 ld hl,#0004 
 out (port_io_bank0),a  ; 28-11-2024 use bank0
 ret
 
;***********  Init ZX SPECTRUM  INTERRUPT  **********

zx_set_interrupt:
 ld a,#C3 ; opcode jmp
 ld hl,#0038 ; interrupt address RST #38
 out (port_io_bank3),a  ; write it
 ld de,zx_int_handler
 ld a,e  ; lsb
 inc hl
 out (port_io_bank3),a  ; write it 
 ld a,d  ; msb
 inc hl
 out (port_io_bank3),a  ; write it  
 ret
 

;***** Initialize ZX SPECTRUM VIDEO ******


;Each line has 32 consecutive addresses starting at the following for each line;
;
;-- first section --
;#4000 4100 4200 4300 4400 4500 4600 4700   textline 0 (8 rows/line) color at #5800 .. #581F
;#4020 4120 ..                       4720                            color at #5820
;#4040 4140 ..
;...
;#40E0 41E0 ..                       47E0   textline 7               color at #58E0 .. #58FF
;
;-- second section --
;#4800 ... 4F00  textline 8  color at  #5900 .. #591F
;#4820 ... 4F20              color at  #5920
;...
;#48E0 ... 4FE0 
;
;-- third section -- 
;#5000 ... 5700  textline 16  color at  #5A00 .. #5A1F
;#5020 ... 5720
;...
;#50E0 ... 57E0  textline 23  color at #5AE0 .. #5AFF
;


zx_pos: dw #4020 ; start position at screen. Leave upper row unused.

zx_attr_table:

 db 0  ; black
 db col_blue   ; 1 blue
 db col_red    ; 2 red
 db col_red+col_blue   ; 3 magenta red+blue
 db col_green          ; 4 green
 db col_blue+col_green ; 5 cyan   blue+green
 db col_red+col_green  ; 6 yellow red+green
 db col_red+col_green+col_blue   ; 7 white
 
zx_video_init:

 if 0
 ld hl,vga_frame   ; now for the first position
; ld a,v_repeat
; out (port_io_bank1),a  ; this will temp. stop the video frame generator 28-11-2024

 ld bc,60000 ; wait until current vga scan is completed, 30-12-2024
zx_dly2: 
 dec bc ; 5
 ld a,b ; 2
 or c  ; 3
 jr nz,zx_dly2 ; 8
 endif
 
 ld hl,#4020
 ld (zx_pos),hl  ; set cursor to 2nd line 
; init end-of-line flags
; first set all flag bytes to zero
 ld hl,zx_bitmap
 xor a
 ld c,24 ; there are 24 textlines
zx_eolclr2:
 ld b,64 ; for clearing 256 values (one text line)
zx_eolclr:
 out (port_io_bank1),a
 ;ld (hl),a  ; also clr normal screen, 17-6-2024
 inc hl 
 out (port_io_bank1),a
 ;ld (hl),a  
 inc hl
 out (port_io_bank1),a
 ;ld (hl),a 
 inc hl
 out (port_io_bank1),a
 ;ld (hl),a  
 inc hl
 nop
 djnz zx_eolclr
 dec c
 jr nz,zx_eolclr2
; now set eol flags
 ld hl,zx_bitmap+32  ; The flag is in the character just after the last char on a line
 ld b,192  ; number of lines
 ld de,32 ; increment value
 ld a,#80
zx_eolset: 
 out (port_io_bank1),a
 add hl,de
 nop
 djnz zx_eolset
; eol flags have been set now.

;Initialize color translation
 ld hl,zx_color_trans
 ld b,0
 ld d,0
zx_ctran:
 ; color attribute, FBPP PIII  F=flash, B=Bright, P=paper(BG), I=Ink (FG)
 ; colors, bit2 green, bit1 red, bit0 blue
 ld a,b
 and 7
 ld e,a 
 ld ix,zx_attr_table
 add ix,de ; index in attr table for FG color
 ld a,(ix) ; get Isetta color
 bit 6,b  ; bright ?
 jr nz,zx_ctran1 ; jp if bright
 or a
 jr z,zx_ctran2 ; for black, dont add gray
; add a,#15  ; add a little gray otherwise the color is too dark, remove 29-11-2024
 jr zx_ctran2
zx_ctran1: 
 ; bright, " remove add it twice" 
 ;add a,(ix) ; add it twice, so 1 will become 3, 4 will become 12 etc
 ;add a,(ix)
 add a,a ;
 add a,a
 add a,a
 add a,a ; shift 4 pos. left
 add a,(ix) ; add original to it 

zx_ctran2:
 out (port_io_bank0),a  ; put in FG translation table 
 ld a,b
 rra
 rra
 rra    ; put paper (BG) in bit2,1 and 0
 and 7
 ld e,a 
 ld ix,zx_attr_table
 add ix,de ; index in attr table for BG color
 ld a,(ix) ; get Isetta color
 bit 6,b  ; bright ?
 jr nz,zx_ctran5 ; jp if bright
 or a
 jr z,zx_ctran6; for black, dont add gray 
 ;add a,#15  ; add a little gray, remove 29-11-2024
 jr zx_ctran6
zx_ctran5: 
 ; bright, " remove add it twice" 
 ;add a,(ix) ; add it twice, so 1 will become 3, 4 will become 12 etc
 ;add a,(ix)
 add a,a ; 
 add a,a
 add a,a
 add a,a ; shift 4 pos. left
 add a,(ix) ; add original to it  
zx_ctran6:
 out (port_io_bank2),a  ; put in BG translation table
 inc hl
 inc b
 jr nz,zx_ctran
; translation table is filled now
; finally, prepare the frame

; frame table add in HL
 ld hl,vga_frame
 inc hl  ; skip the prepare command
 ld c,#58  ; first MSB for video attribute
 ld de,#4000 ; bitmap start address
; line addr in DE 
zx_fr_third:
 push de
 ld b,8  ; text line, 8 rows
zx_fr_textline: 
 ld a,v_spectrum        ; command for zx_spectrum video line
 out (port_io_bank1),a  ; command
 ld a,h  ; 30-8-2025
 sub #04 ; 30-8-2025 set hl from #700 to #300
 ld h,a  ; 30-8-2025
 
 ld a,e
 out (port_io_bank0),a  ; bitmap data lsb  ; 15-11-2024 bank0 was bank 4
 ld a,d
 out (port_io_bank2),a  ; bitmap data msb; 15-11-2024 bank2 was bank 5

 ld a,c
 out (port_io_bank1),a  ; attr addr msb   // 27-11-2024 bank1 instead of 6
 inc hl
 out (port_io_bank1),a  ; attr addr msb also for next line
 ; each line is displayed twice, so next line has same data;
 ld a,e
 out (port_io_bank0),a  ; bitmap data lsb ; 15-11-2024 bank0 was bank 4
 ld a,d
 out (port_io_bank2),a  ; bitmap data msb; 15-11-2024 bank2 was bank 5
 ld a,h  ; 30-8-2025
 add a,#04 ; 30-8-2025 set hl from #300 back to #700
 ld h,a  ; 30-8-2025 
 ld a,v_spectrum        ; command for zx_spectrum video line, 30-8-2025 here
 out (port_io_bank1),a  ; command 
 inc hl
 ; end of 2nd line
 inc d      ; next row, address #100 higher
 djnz zx_fr_textline
 
 pop de
 ld a,e
 add a,#20
 ld e,a ; next text line
 jr nz,zx_fr_third  ; jp back if this 'third' not yet complete
 ld a,d
 add a,#08
 ld d,a   ; for next third, bitmap address #0800 higher
 inc c  ; next msb for attr table
 ld a,c
 cp #5B  ; if msb is #5B, we're too far, stop in that case
 jr nz,zx_fr_third
 ; 2 x 192 lines have been defined now
 ld a,v_end             ; command for end of visible display
 out (port_io_bank1),a  ; command 
 inc hl
; add 32 blank lines, to be sure to overwrite the end of the previous display frame
 ld b,32   ; although 16 blank lines should be sufficient
 ld a,v_blank
zx_fr_blnk:
 out (port_io_bank1),a  ; 16 blank commands
 nop
 inc hl
 djnz zx_fr_blnk
 ret
 
 
;********     M A I N     ********* 

zx_main:

 di
 
;first, copy the keyboard handler and font to the ZX Spectrum ROM location in HW bank3

 ld de,#400 ; source
 ld hl,kbhandler ; destination, #3800
 ld bc,font_end-kbhandler
zx_mainloop1:
 ld a,(de)
 out (port_io_bank3),a ; store byte at (hl) in bank3
 inc hl
 inc de 
 dec bc 
 ld a,b 
 or c 
 jr nz,zx_mainloop1

; second, copy the application to its destination address in HW bank 3
 
 ld de,font_end - kbhandler + #400; source
 ld hl,application ; destination
 ld bc,application_end - application
zx_mainloop2:
 ld a,(de)
 out (port_io_bank3),a ; store byte at (hl) in bank3
 inc hl
 inc de 
 dec bc 
 ld a,b 
 or c 
 jr nz,zx_mainloop2
 
; 15-9-2025 add clearing of sound buffer
 ld hl,#0480
 ld b,150 ; amount of bytes to clear
 ld a,#7E ; this is the zero value
snd_loop:
  out (port_io_bank0),a 
  inc hl
  djnz snd_loop 
  
 ld a,1
 out (port_out_sound_mode),a ; 1-9-2025 select ZX Spectrum 1-bit sound mode
 
 ; overwrite the v_sound instruction in the frame, at 3 positions
 ; The v_sound would not write a speaker sample, so
 ; replace it by a v_blank
 ld a,v_blank
 ld hl,#089c ; decimal 412 from start (8-7-2025 at new video frame addr #700)
 out (port_io_bank1),a 
 ld l,#D1  ; #04D1, dec 465 from start
 out (port_io_bank1),a 
 ld l,#F4  ; #04F4, dec 500 from start
 out (port_io_bank1),a     
 
 xor a
 out (port_out_system),a  ; 22-12-2025 set system to default
 
 
if use_interrupt =1
 ei 
else
 di
endif 
 
 
 ld a,#C0
 db #D3,#1E ; bankswitch to HW page 3 in all 4 blocks
 dw appl_run ; and jump to appl_run in bank3
 

 
; In the cp/m memory map there is application space from 0x0100 to 0xDC00.
; But a ZX Spectrum application can start at 0x4000 and go up to 0xFFFF
; Therefore, the application is shifted down, to fit in the CP/M memory map
; We now fill up to the 1KB boundary and then leave a 'hole' in the memory, to do this downshift.

 ds #400-$   
 

;*******  Z X   SPECTRUM  KEYBOARD HANDLING  ********

; This must be on a 256-byte boundary.
; The MSB of the address is stored as Isetta variable.
; When the keyboard is read in an IN instruction, a call to this address is done.

  org kbhandler
 
  call zx_readkey  ; reads a key from PS/2. Delivers ASCII code.

ascii_to_zx:
  ; will now put this in the Spectrum scantable
  push hl
  push de
  push bc  
  ;in a,(port_in_scancode)  ; key pressed ?
  or a
  ;jr z,k_readtable ; no message, just read from the table
  jp z,k_timer 
  ; ascii in ACC
  ld (kb_ascii),a
;  cp "Q"
;  jr z,k_exit
;  cp #1B  ; esc
;  jp z,0  ; ESC will take you back to ZXOS operating system

  ld hl,k_count
  ld (hl),100  ; init counter for releasing the key 
  ld e,gr6
  ld d,bit0 ; gr6, bit0 is ENTER
  and #7F  ; clear ascii bit7
  cp #0D
  jr z,k_fnd
  cp #20
  jr c,k_timer  ; do nothing for other control codes
  cp #60
  jr c,k_upper
  sub #20 ; lower -> upper
  cp "Z"+1
  jr c,k_case  ; jp if a letter
  ; it is one of  { } | ~ 
  add a,5  ; put it directly behind [ \ ] ^ _
  jr k_case
k_upper:
  cp "Z"+1
  jr nc,k_case ; jp if not a letter
  cp "A"
  jr c,k_case  ; below "A", no upper case
  ld hl,k_row7+1
  bit 1,(hl)   ; was 'sym' activated (0), in that case no upper case
  jr z,k_case
  ld hl,k_row3+1
  res 0,(hl)  ; activate shift to indicate upper case (active low)
k_case:
  ; search the ascii in a table. table from #20 - #5F
  sub #20  ; the table starts at #20
  ld e,a
  ld d,0
  ld hl,asc_to_zx
  add hl,de
  add hl,de  ; add index to hl
  ;call prhl
  ld e,(hl) ; group
  inc hl
  ld d,(hl) ; bit
  bit 7,e
  jr z,k_fnd
  ; if bit7 of group is set, it indicates a symbol
  ld hl,k_row7+1
  res 1,(hl)  ; activate SYM key
  ld a,d    ; instead of the bit, it is another ascii
  jr k_case
  
k_fnd:
  ; now group in D, bit in E
  ld hl,key_table+1
  ld c,d ; save bit nr
  ld d,0
  add hl,de
  add hl,de
;  call prhl ;; debug  print table position
  ; now hl points to correct group byte
  ; put bit in A
  ld a,c
;  call prbyte ; debug print bit
  cpl
  and (hl)
  ld (hl),a ; reset the proper bit in the table
;  call prbyte ; debug print new value
;  ld a,#20
;  call chout
k_readtable: 

 
 ; this is a new version, that can read several rows at the same time
  ld hl,key_table
  ld e,8
  ld d,#FF ; prepare result
  ; B contains the row mask  (FE, F7 etc)
k_search:
  ld a,b
  or (hl)  ; this will leave non-FF if this is a corresponding row
  inc hl
  inc a ; was it FF
  jr z,k_next ; yes it was FF, this row was not selected
  ld a,(hl) ; get it's data reading
  and d      ; AND it with what we already have
  ld d,a      ;  and store it back
k_next:  
  inc hl
  dec e
  jr nz,k_search
  ld a,d ; get the value
k_99:  
  pop bc
  pop de
  pop hl
  ret  
  ; end new version

 
k_timer:
  ld hl,k_toggle
  in a,(port_io_reg_vga_ptl)  ; pointer to frame
  cp #BF
  jr nc,k_timer8
  bit 0,(hl)
  jr z,k_pulse
  set 0,(hl)
  jr k_readtable
k_timer8:
  ld (hl),0
k_timer9:
  jr k_readtable
 
k_toggle: db 0 

k_count: db 100

kb_ascii db 0

k_pulse:
  ; an impulse was detected in the frame sequence, that
  ; means that another frame has passed by
 set 0,(hl)
 inc hl  ; point to k_count 
 dec (hl)
 jr nz,k_readtable
 ; endcount reached
 ld (hl),100 ; new endcount
 ; now clear all active bits
 ld hl,key_table
 ld b,8
k_clr_loop:
 inc hl
 ld (hl),#FF ; set all keys inactive
 inc hl
 djnz k_clr_loop
 jr k_readtable
 
 
gr0 equ 0
gr1 equ 1
gr2 equ 2
gr3 equ 3
gr4 equ 4
gr5 equ 5
gr6 equ 6
gr7 equ 7

;*** ZX Spectrum keyboard organization ***
;
;                 -- left -   - right -
;Bit numbers      0 1 2 3 4   4 3 2 1 0
;
;Upper row  0 F7  1 2 3 4 5   6 7 8 9 0  EF 4  
;Next row   1 FB  q w e r t   y u i o p  DF 5 
;Next row   2 FD  a s d f g   h j k l <  BF 6  "<" is ENTER
;Lower row  3 FE  ^ z x c v   b n m * _  7F 7   ^=shift, *=sym, _=space
;
; keys are organized in 8 groups of 5. The groups are numbered 0 - 7,
; group 0-3 at the left side, and group 4 - 7 at the right side of the keyboard.
;
;

asc_to_zx:   ; table to convert ASCII code to group + bitnr
; symbols 20 - 3F
; note that the INDEX is according to ASCII
 db gr7, bit0  ; space
 db #80,"1"  ; !
 db #80,"P"  ; "
 db #80,"3"  ; #
 db #80,"4"  ; #
 db #80,"5"  ; %
 db #80,"6"  ; & 
 db #80,"7"  ; '
 db #80,"8"  ; (
 db #80,"9"  ; )
 db #80,"B"  ; *
 db #80,"K"  ; +
 db #80,"N"  ; ,
 db #80,"J"  ; -
 db #80,"M"  ; .
 db #80,"V"  ; /
; digits 
 db gr4, bit0 ; 0
 db gr0, bit0 ; 1
 db gr0, bit1 ; 2
 db gr0, bit2 ; 3
 db gr0, bit3 ; 4
 db gr0, bit4 ; 5
 db gr4, bit4 ; 6
 db gr4, bit3 ; 7
 db gr4, bit2 ; 8
 db gr4, bit1 ; 9
 db #80,"Z"  ; 
 db #80,"O"  ; ;
 db #80,"R"  ; <
 db #80,"L"  ; =
 db #80,"T"  ; >
 db #80,"C"  ; ? 

; letters 40-5F
 db #80,"2"   ; @
 db gr2, bit0 ; a  Note that this should be capitals
 db gr7, bit4 ; b
 db gr3, bit3 ; c
 db gr2, bit2 ; d
 db gr1, bit2 ; e
 db gr2, bit3 ; f
 db gr2, bit4 ; g
 db gr6, bit4 ; h
 db gr5, bit2 ; i
 db gr6, bit3 ; j
 db gr6, bit2 ; k
 db gr6, bit1 ; l
 db gr7, bit2 ; m
 db gr7, bit3 ; n
 db gr5, bit1 ; o
 db gr5, bit0 ; p
 db gr1, bit0 ; q
 db gr1, bit3 ; r
 db gr2, bit1 ; s
 db gr1, bit4 ; t
 db gr5, bit3 ; u
 db gr3, bit4 ; v
 db gr1, bit1 ; w
 db gr3, bit2 ; x
 db gr5, bit4 ; y
 db gr3, bit1 ; z
 db #80,"Y"  ; [
 db #80,"D"  ; \
 db #80,"U"  ; ]
 db #80,"H"  ; ^
 db #80,"0"  ; _  
 ; now codes follow for { | } ~  --->  f s g a DEL
 db #80,"F"  ; {
 db #80,"S"  ; |  
 db #80,"G"  ; }
 db #80,"A"  ; ~
 db 0,0      ; DEL

key_table:
 ; this has two bytes for every key group
 ; the first byte is the F7, FB, FD etc code of the group, the second
 ; byte is the actual state of the keys in that group, 0=pressed.
 db #F7,#FF  ;  
 db #FB,#FF
 db #FD,#FF  
k_row3:
 db #FE,#FF
 db #EF,#FF
 db #DF,#FF
 db #BF,#FF  
k_row7:
 db #7F,#FF
 db #FF     ; for the 'not found' case
 

 

;*****************************************************
;
; PS/2 keyboard decoder
;
; 30-5-2024
;
; Conversion from PS/2 keyboard scancodes to character codes.
;
; For the Isetta TTL computer.
;
; Character scancodes from;   (uses US keyboard version)
;
; webdocs.cs.ualberta.ca/~amaral/courses/329/labs/scancodes.html
;

; PS/2 keyboard decoder
;
; bit flags;  ( at (hl) )

rk_fl_capslock equ 0
rk_fl_num     equ 1
rk_fl_ctrl    equ 2
rk_fl_alt     equ 3
rk_fl_shift   equ 4
rk_fl_scroll  equ 5  ; unused
rk_fl_e0      equ 6
rk_fl_release equ 7

FUNCTION equ #81
LETTER  equ #82
MODIFIER equ #83
SINGLE  equ #84

; Now follow the result codes for navigation keys and function keys.
; There is no escape sequence used, but that could be easily changed.

KB_NAV_UP equ #81
KB_NAV_DN equ #82
KB_NAV_RIGHT equ #83
KB_NAV_LEFT equ #84
KB_NAV_END equ #86
KB_NAV_HOME equ #88
KB_NAV_INSERT equ #89
KB_NAV_DELETE equ #8A
KB_NAV_PGUP equ #8B
KB_NAV_PGDN equ #8C

KB_FN_F1 equ #91
KB_FN_F2 equ #92
KB_FN_F3 equ #93
KB_FN_F4 equ #94
KB_FN_F5 equ #95
KB_FN_F6 equ #96
KB_FN_F7 equ #97
KB_FN_F8 equ #98
KB_FN_F9 equ #99
KB_FN_F10 equ #9A
KB_FN_F11 equ #9B
KB_FN_F12 equ #9C


rk_scancode: db 0
rk_prefix: db 0
rk_flags: db 0
rk_mode: db 0    ; start with 0. Becomes non-zero when a PS/2 key is detected.
    ; As soon as a PS/2 key is detected there will be no more output to RPi


; function ZX_READKEY
; input; nothing
; output;
;    A = 0  if no key was pressed.
;    If A !=0, it is the ASCII code of the pressed key. This can
;    be a full 8-bit code, so also characters 80 - FF can be represented
;    The navigation keys (arrow keys, home, etc) are in the #81-#8C range.
;    The 12 function keys are in the #91-#9C range.
;
; The key input can come from a ps/2 keyboard scancode but also from a host
; system (Raspberry Pi) ASCII code (via a synchronous serial link).
; scancodes and serial link codes come from the same input port , #31.
; When the serial link is used, the ps/2 keyboard should be disconnected.


zx_readkey:

 in a,(port_in_scancode)   ; get scancode
 or a
 ret z         ; return if nothing available

 push bc
 push hl
 ld hl,rk_flags ; addr of flags

 cp #07  ; reverse E0
 jr nz,rk_1
 set 6,(hl) ; E0 received, set E0 flag
 ;ld a,#E0
 ;ld (rk_prefix),a
 ;jr rk_3 ; 7-12-2025 this is wrong !!
 jr rk_1A  ; 7-12-2025

rk_1:
 cp #0F   ; reverse F0
 jr nz,rk_2
 ; a Key-up event was received
 ld (rk_mode),a  ; this will switch RPi communication OFF
 set 7,(hl)  ; set release flg (key-up event)
rk_1A: 
 xor a  ; return without result
rk_5: 
 pop hl
 pop bc
 ret

rk_2:

 ld hl,rk_flags
 bit 6,(hl) ; check E0 flag

 ld h,a  ; code to H
 
 ld a,#E0
 jr nz,rk_2A
 xor a 
rk_2A: 
 ld (rk_prefix),a ; 7-12-2025
  
; ld a,(rk_mode)
; or a
; ld a,h
; jr z,rk_5  ; if z, we suppose input came from RPi

 ; now we suppose input from PS/2 keyboard

 ; go reverse the bits

 add hl,hl ; shift bit out H from left side
 rra    ; shift into A at left side
 add hl,hl
 rra
 add hl,hl
 rra
 add hl,hl
 rra
 add hl,hl
 rra
 add hl,hl
 rra
 add hl,hl
 rra       
 add hl,hl
 rra     ; 8 bits reversed 

 ld (rk_scancode),a  ; save scancode for debugging

 cp #83 ; special handling for scancode #83 because it's outside the 00-7F range
 jr nz,rk_no83
 dec a   ; replace #83 by scancode 2 (that was not used)
rk_no83: 

 and #7F  ; reset bit7
 
 rla   ; code * 2
 ;call prbyte ; DEBUG, print scancode (times 2 !!)
 
 ld bc,scancode_table
 ld l,a

; ld a," "
; call chout ; debug, output formatting
 
 ld h,0
 add hl,bc ; now HL is index in table, 2 bytes per entry
 ld a,(hl) ; first byte from table
 inc hl
 ld c,(hl) ; second byte from table 
 ld hl,rk_flags ; re-load addr of flags

 cp FUNCTION
 jr nz,rk_fnd2
 ; This is one of the function keys F1 - F12
 ld a,c
 jr rk_end ; return 2nd byte as code


rk_fnd2:
 cp LETTER
 jr nz,rk_fnd3
 ; This is an ordinary lower-case letter
 ; But it will turn to uppercase when SHIFT is pressed or CAPSLOCK is active.
 ; When CTRL is pressed, it gives the corresponding control code.
 ld a,c
 bit 2,(hl)  ; check ctrl flag
 jr z,rk_noctrl
 and #1F  ; it's a CTRL char !
 jr rk_end
rk_noctrl:
 bit 4,(hl)  ; check SHIFT flag
 jr nz,rk_cap
 bit 0,(hl)  ; check capslock flag
 jr z,rk_end
rk_cap:
 xor #20  ; make it capital

rk_end:
 ; end handling for regular chars.
 bit 7,(hl)   ; check release flag
 jr z,rk_99   ; if no release flag, normal key press
 ;jr z,rk_80   ; if no release flag, normal key press 
 
  ; for ZX Spectrum keyboard; 
 ld hl,k_row7+1
 ld (hl),#FF  ; release keys in the spacebar group
 cp " "  ; was space released ?
 jr z,rk_4  ; in that case, jump (only release the spacebar group keys)
 ld hl,key_table
 ld b,8
rk_3:
 inc hl
 ld (hl),#FF ; set all keys inactive
 inc hl
 djnz rk_3
rk_4: 
 ld hl,rk_flags ; re-load addr of flags
  ; end spectrum keyboard
 
rk_none:
 res 7,(hl)  ; reset release flag
 xor a  ; result zero, no character
rk_99:  
 res 6,(hl) ; clear E0 flag
 pop hl
 pop bc
 ret
 
rk_80:     ; 23-12-2025 new section to put ascii char in #5C08 (23560)
 res 6,(hl) ; clear E0 flag
 ld (#5C08),a 
 ld hl,#5C3B  ; ZX Spectrum flags at 23611 
 set 5,(hl)   ; indicate a new ascii is present
 pop hl
 pop bc 
 ret
 
 
rk_fnd3:
 cp MODIFIER
 jr nz,rk_fnd4
 ; This is a MODIFIER key. Something like shift, ctrl, alt, capslock, numlock
 ld a,c
 cp 1
 jr z,rk_capslock
 cp 2
 jr z,rk_capslock ; same for numlock
 bit 7,(hl) ; check release flag
 jr nz,rk_mod_release
 or (hl)  ; set indicated modifier
rk_flgsav:
 ld (hl),a
 jr rk_none
rk_mod_release:
 xor #FF
 and (hl)
 ;ld (hl),a
 jr rk_flgsav
 
rk_capslock:
 bit 7,(hl)  ; check release flag
 jr nz,rk_none   ; if release flag, no further action
 xor (hl)     ; A is still 1 (or 2) so this toggles bit 0 in E (capslock) or bit 1 (numlock)
 ;ld (hl),a
 jr rk_flgsav

rk_fnd4:
 cp SINGLE
 jr z,rk_2nd  ; This is a SINGLE key. It always gives the same code.

rk_fnd5:
 bit 7,a
 jr z,rk_shift
 ; This is a KEYPAD key. Also used for navigation keys.
 ; The resulting code depends on the NUMLOCK state (toggled with numlock key)
 ; first byte has bit7=1. This is mainly for keypad codes.
 bit 6,(hl)  ; check E0 flag
 jr nz,rk_2nd   ; jp if E0 prefix was seen
 and #7F ;reset bit7
 bit 1,(hl)  ; check NUM flag
 jr nz,rk_end    ; jp if in NUM mode, use 2nd byte
 ; if not in NUM mode, also use the E0 code of this key;
rk_2nd:
 ld a,c
 or a ; but also '0', unimplemented code can be here, check it. 7-12-2025
 jr z,rk_zero 
 jr rk_end ; second byte is the result code. 

rk_shift:
 ; This is a SHIFTER key. It gives different code when SHIFT is pressed.
 ; first byte has bit7=0. This is the shifted code.
 ; the second byte (in c) is the unshifted code.
 
 or a ; but also '0', unimplemented code can be here, check it. 7-12-2025
 jr z,rk_zero
 
 bit 4,(hl)  ; check SHIFT flag

 jr nz,rk_end  ; jp if shifted (A = first byte, is result)

 jr rk_2nd  ; jp if not shifted, second byte is result

rk_zero:
 ld a,"?"
 jr rk_end


scancode_table:

; PS/2 Scancode table 2, starting with code 00 - 0F. Two bytes per code.

 db 0,0                ; code 0
 db FUNCTION,KB_FN_F9  ; code 1  F9
 db FUNCTION,KB_FN_F7  ; code 2   Scancode #83 comes here
 db FUNCTION,KB_FN_F5  ; code 3  F5
 db FUNCTION,KB_FN_F3  ; code 4  F3
 db FUNCTION,KB_FN_F1  ; code 5  F1
 db FUNCTION,KB_FN_F2  ; code 6  F2
 db FUNCTION,KB_FN_F12 ; code 7  F12
 db 0,0                ; code 8 
 db FUNCTION,KB_FN_F10 ; code 9   F10
 db FUNCTION,KB_FN_F8  ; code 0A  F8
 db FUNCTION,KB_FN_F6  ; code 0B  F6
 db FUNCTION,KB_FN_F4  ; code 0C  F4
 db SINGLE,#09 ; code 0D  09 TAB
 db #7E,#60   ; code 0E shifted 7E ~ unshifted 60 `
 db 0,0         ; code 0F 
 
; Code values 10 - 1F

 db 0,0            ; code 10
 db MODIFIER,#08  ; code 11  r-alt, l-alt(E0)
 db MODIFIER,#10  ; code 12  shift, prntscr(E0)
 db 0,0            ; code 13 
 db MODIFIER,#04  ; code 14  r-ctrl, l-ctrl(E0)
 db LETTER,"q"     ; code 15  Q
 db "!","1"        ; code 16   shifted  !  unshifted 1
 db 0,0            ; code 17 
 db 0,0            ; code 18 
 db 0,0            ; code 19 
 db LETTER,"z"     ; code 1A  Z
 db LETTER,"s"     ; code 1B  S
 db LETTER,"a"     ; code 1C  A
 db LETTER,"w"     ; code 1D  W 
 db "@","2"        ; code 1E   shifted  @  unshifted 2
 db SINGLE,#00    ; code 1F  l-WIN(E0)  no result code yet

; Code values 20 - 2F

 db 0,0            ; code 20
 db LETTER,"c"     ; code 21  C
 db LETTER,"x"     ; code 22  X
 db LETTER,"d"     ; code 23  D
 db LETTER,"e"     ; code 24  E  
 db "#","4"        ; code 25   shifted  #  unshifted 4 
 db "#","3"        ; code 26   shifted  #  unshifted 3 
 db SINGLE,#00    ; code 27  r-WIN(E0)  no result code yet 
 db 0,0            ; code 28
 db SINGLE,#20    ; code 29  space 
 db LETTER,"v"     ; code 2A  V
 db LETTER,"f"     ; code 2B  F
 db LETTER,"t"     ; code 2C  T
 db LETTER,"r"     ; code 2D  R  
 db "%","5"        ; code 2E   shifted  %  unshifted 5  
 db SINGLE,#00    ; code 2F  apps  no result code yet  
 
; Code values 30 - 3F

 db 0,0            ; code 30
 db LETTER,"n"     ; code 31  N
 db LETTER,"b"     ; code 32  B
 db LETTER,"h"     ; code 33  H
 db LETTER,"g"     ; code 34  G  
 db LETTER,"y"     ; code 35  Y  
 db "^","6"        ; code 36   shifted  ^  unshifted 6
 db 0,0            ; code 37 
 
 db 0,0            ; code 38
 db 0,0            ; code 39 
 db LETTER,"m"     ; code 3A  M
 db LETTER,"j"     ; code 3B  J
 db LETTER,"u"     ; code 3C  U  
 db "&","7"        ; code 3E   shifted  &  unshifted 7
 db "*","8"        ; code 3E   shifted  *  unshifted 8
 db 0,0            ; code 3F 
 
; Code values 40 - 4F 
 
 db 0,0            ; code 40 
 db "<",","        ; code 41   shifted  <  unshifted ,
 db LETTER,"k"     ; code 42  K
 db LETTER,"i"     ; code 43  I
 db LETTER,"o"     ; code 44  O  
 db ")","0"        ; code 45   shifted  )  unshifted 0
 db "(","9"        ; code 46   shifted  (  unshifted 9 
 db 0,0            ; code 47 
 
 db 0,0            ; code 48 
 db ">","."        ; code 49   shifted  >  unshifted . 
 db "?","/"        ; code 4A   shifted  ?  unshifted / (also E0, keypad)
 db LETTER,"l"     ; code 4B  L 
 db ":",";"        ; code 4C   shifted     unshifted ; 
 db LETTER,"p"     ; code 4D  P 
 db "_","-"        ; code 4E   shifted  _  unshifted -  
 db 0,0            ; code 4F  
 
; Code values 50 - 5F  
 
 db 0,0            ; code 50
 db 0,0            ; code 51  
 db #22,#27      ; code 52   shifted  "  unshifted '  
 db 0,0            ; code 53  
 db "{","["        ; code 54   shifted  {  unshifted [   
 db "+","="        ; code 55   shifted  +  unshifted =    
 db 0,0            ; code 56
 db 0,0            ; code 57  
 
 db MODIFIER,#01  ; code 58  capslock
 db MODIFIER,#10  ; code 59  r-shift
 db SINGLE,#0D    ; code 5A  enter (also E0) 
 db "}","]"        ; code 5B   shifted  }  unshifted ] 
 db 0,0            ; code 5C  
 db "|",#5C       ; code 5D   shifted  |  unshifted \   
 db 0,0            ; code 5E
 db 0,0            ; code 5F  
 
; Code values 60 - 6F  

 db 0,0            ; code 60 
 db 0,0            ; code 61 
 db 0,0            ; code 62 
 db 0,0            ; code 63 
 db 0,0            ; code 64 
 db 0,0            ; code 65 
 db SINGLE,#08    ; code 66  backspace
 db 0,0            ; code 67  

 db 0,0            ; code 68 
 db #80+"1",KB_NAV_END    ; code 69 keypad 1, END 
 db 0,0                    ; code 6A 
 db #80+"4",KB_NAV_LEFT   ; code 6B keypad 4, L-ARROW
 db #80+"7",KB_NAV_HOME   ; code 6C keypad 7, HOME
 db 0,0            ; code 6D 
 db 0,0            ; code 6E 
 db 0,0            ; code 6F  
 
; Code values 70 - 7F   (Navigation outside keypad will also work)

 db #80+"0",KB_NAV_INSERT  ; code 70 keypad 0, INSERT
 db #80+".",KB_NAV_DELETE  ; code 71 keypad ".", DELETE
 db #80+"2",KB_NAV_DN      ; code 72 keypad 2, D-ARROW
 db #80+"5",0              ; code 73 keypad 5
 db #80+"6",KB_NAV_RIGHT   ; code 74 keypad 6, R-ARROW
 db #80+"8",KB_NAV_UP      ; code 75 keypad 8, U-ARROW
 db SINGLE,#1B    ; code 76  esc 
 db MODIFIER,#02  ; code 77  numlock 

 db FUNCTION,KB_FN_F11     ; code 78  F11
 db SINGLE,"+"             ; code 79 keypad +
 db #80+"3",KB_NAV_PGDN   ; code 7A keypad 3, PGDN
 db SINGLE,"-"             ; code 7B keypad -
 db SINGLE,"*"             ; code 7C keypad *  (prntscr)
 db #80+"9",KB_NAV_PGUP   ; code 7D keypad 9, PGUP
 db MODIFIER,#20  ; code 7E  scroll lock  
 db 0,0            ; code 7F  
 
 ; Done with the table !!

 ; end of ps/2 keyboard decoder
 
cpir72:     ; patch for Manic Miner 
  LD BC,72                ; data of the background, floor, crumbling floor,
  
cpirlp:
 cp (hl) 
 inc hl
 dec bc
 ret z
 inc c
 dec c
 jr nz,cpirlp ; jp if c is nz
 inc b
 djnz cpirlp ; test if B is zero, (inc then dec) without changing A 
 ; now force an NZ , how ?
 ; djnz doesn't influence Z flag. Z will be cleared because inc b result was 1.
 ret
 
;***********  ZX SPECTRUM  INTERRUPT HANDLER  **********

zx_int_handler :
   push af
   push hl
   ld hl,(#5C78)     ; Fetch the first two bytes at FRAMES1.
   inc hl            ; Increment lowest two bytes of counter.
   ld (#5C78),hl     ; Place back in FRAMES1.

   call zx_readkey    ; if a key pressed, return ascii value, otherwise 0
   or a 
   jr z,zx_int_handler9
   
   ld (#5C08),a 
   ld hl,#5C3B  ; ZX Spectrum flags at 23611 
   set 5,(hl)   ; indicate a new ascii is present   
   
   call ascii_to_zx
   
zx_int_handler9:   
   pop hl
   pop af
   ret   
 
;***********  ZX SPECTRUM  FONT  **********  
  
; ZX spectrum font is at 3D00 - 3FFF
; just as in the original Spectrum 

;org #3D00 
      ds #3D00-$ 
 
 fontsrc:
	db #00,#00,#00,#00,#00,#00,#00,#00 ;  
	db #10,#10,#10,#10,#10,#00,#10,#00 ; !
	db #6c,#6c,#90,#00,#00,#00,#00,#00 ; "
;	db #00,#24,#7e,#24,#24,#7e,#24,#00 ; #
	db #00,#28,#7c,#28,#28,#7c,#28,#00 ; # new (two empty pos at end)
	db #10,#3c,#50,#38,#14,#78,#10,#00 ; #
	db #40,#a4,#48,#10,#24,#4a,#04,#00 ; %
	db #30,#48,#48,#32,#4a,#44,#3a,#00 ; & 
	db #18,#18,#20,#00,#00,#00,#00,#00 ; '
	db #08,#10,#20,#20,#20,#10,#08,#00 ; (
	db #20,#10,#08,#08,#08,#10,#20,#00 ; )
	db #00,#10,#54,#38,#54,#10,#00,#00 ; *
	db #00,#10,#10,#7c,#10,#10,#00,#00 ; +
	db #00,#00,#00,#00,#00,#18,#18,#20 ; ,
	db #00,#00,#00,#7c,#00,#00,#00,#00 ; -
	db #00,#00,#00,#00,#00,#18,#18,#00 ; .
;   db #02,#04,#08,#10,#20,#40,#80,#00 ; /
    db #00,#04,#08,#10,#20,#40,#80,#00 ; /  new (two empty pos at end)
	db #38,#44,#44,#44,#44,#44,#38,#00 ; 0
	db #30,#50,#10,#10,#10,#10,#10,#00 ; 1
	db #78,#04,#04,#18,#20,#40,#7c,#00 ; 2
	db #7c,#04,#08,#18,#04,#04,#78,#00 ; 3
	db #10,#10,#20,#28,#48,#7c,#08,#00 ; 4
	db #7c,#40,#40,#78,#04,#04,#78,#00 ; 5
	db #08,#10,#20,#78,#44,#44,#38,#00 ; 6
	db #7c,#04,#08,#10,#20,#20,#20,#00 ; 7
	db #38,#44,#44,#38,#44,#44,#38,#00 ; 8
	db #38,#44,#44,#3c,#08,#10,#20,#00 ; 9
	db #00,#18,#18,#00,#00,#18,#18,#00 ; 
	db #00,#18,#18,#00,#00,#18,#18,#20 ; ;
	db #00,#0c,#30,#40,#30,#0c,#00,#00 ; <
	db #00,#00,#7e,#00,#7e,#00,#00,#00 ; =
	;db #00,#30,#0c,#02,#0c,#30,#00,#00 ; >
	db #00,#60,#18,#04,#18,#60,#00,#00 ; >  shift pixels one bit to the left	
	db #38,#44,#04,#08,#10,#00,#10,#00 ; ?
	
	
	db #7c,#82,#ba,#aa,#be,#80,#7c,#00 ; @
	;db #7c,#82,#ba,#aa,#be,#80,#7c,#00 ; @  remove the middle column	TODO
	db #18,#18,#24,#24,#3c,#42,#42,#00 ; A
	db #78,#44,#44,#78,#44,#44,#78,#00 ; B
	db #18,#24,#40,#40,#40,#24,#18,#00 ; C
	db #70,#48,#44,#44,#44,#48,#70,#00 ; D
	db #7c,#40,#40,#78,#40,#40,#7c,#00 ; E
	db #7c,#40,#40,#78,#40,#40,#40,#00 ; F
	db #18,#24,#40,#4c,#44,#24,#1c,#00 ; G
	db #44,#44,#44,#7c,#44,#44,#44,#00 ; H
	db #38,#10,#10,#10,#10,#10,#38,#00 ; I
	db #04,#04,#04,#04,#24,#24,#18,#00 ; J
	db #44,#48,#50,#60,#50,#48,#44,#00 ; K
	db #40,#40,#40,#40,#40,#40,#7c,#00 ; L
	db #44,#6c,#54,#54,#44,#44,#44,#00 ; M
	db #64,#64,#54,#54,#54,#4c,#4c,#00 ; N
	db #18,#24,#42,#42,#42,#24,#18,#00 ; O
	db #78,#44,#44,#78,#40,#40,#40,#00 ; P
	db #18,#24,#42,#42,#4a,#24,#1a,#00 ; Q
	db #78,#44,#44,#78,#50,#48,#44,#00 ; R
	db #38,#44,#40,#38,#04,#44,#38,#00 ; S
	db #7c,#10,#10,#10,#10,#10,#10,#00 ; T
	db #44,#44,#44,#44,#44,#44,#38,#00 ; U
	db #44,#44,#44,#44,#28,#28,#10,#00 ; V
	db #44,#44,#44,#54,#54,#28,#28,#00 ; W
	db #44,#44,#28,#10,#28,#44,#44,#00 ; X
	db #44,#44,#28,#10,#10,#10,#10,#00 ; Y
	db #7c,#04,#08,#10,#20,#40,#7c,#00 ; Z
	db #38,#20,#20,#20,#20,#20,#38,#00 ; [
	db #80,#40,#20,#10,#08,#04,#02,#00 ; \
	;db #80,#80,#40,#20,#10,#08,#04,#00 ; \  shift it one bit down	(7-12-2025 no longer shifted)
	
	db #38,#08,#08,#08,#08,#08,#38,#00 ; ]
	db #10,#28,#28,#44,#00,#00,#00,#00 ; ^
	; db #00,#00,#00,#00,#00,#00,#00,#ff ; _
	db #00,#00,#00,#00,#00,#00,#00,#fc ; _  clear lowest 2 bits
	
	
	;db #08,#14,#10,#78,#20,#20,#7c,#00 ; £  not use pound sign here.
	db #00,#00,#00,#00,#00,#00,#00,#00 ; unused, space for #60
	db #00,#00,#38,#04,#3c,#44,#3c,#00 ; a
	db #40,#40,#58,#64,#44,#44,#78,#00 ; b
	db #00,#00,#38,#44,#40,#44,#38,#00 ; c
	db #04,#04,#34,#4c,#44,#44,#3c,#00 ; d
	db #00,#00,#38,#44,#7c,#40,#3c,#00 ; e
	db #0c,#10,#10,#3c,#10,#10,#10,#00 ; f
	db #00,#00,#3c,#44,#44,#3c,#44,#38 ; g
	db #40,#40,#58,#64,#44,#44,#44,#00 ; h
	db #08,#00,#18,#08,#08,#08,#08,#00 ; i
	db #08,#00,#18,#08,#08,#08,#08,#30 ; j
	db #40,#40,#44,#48,#70,#48,#44,#00 ; k
	db #20,#20,#20,#20,#20,#20,#18,#00 ; l
	db #00,#00,#68,#54,#54,#54,#54,#00 ; m
	db #00,#00,#58,#64,#44,#44,#44,#00 ; n
	db #00,#00,#38,#44,#44,#44,#38,#00 ; o
	db #00,#00,#78,#44,#44,#78,#40,#40 ; p
	db #00,#00,#3c,#44,#44,#3c,#04,#04 ; q
	db #00,#00,#58,#64,#40,#40,#40,#00 ; r
	db #00,#00,#3c,#40,#38,#04,#78,#00 ; s
	db #20,#20,#78,#20,#20,#20,#18,#00 ; t
	db #00,#00,#44,#44,#44,#4c,#34,#00 ; u
	db #00,#00,#44,#44,#28,#28,#10,#00 ; v
	db #00,#00,#44,#44,#54,#6c,#44,#00 ; w
	db #00,#00,#44,#28,#10,#28,#44,#00 ; x
	db #00,#00,#44,#44,#28,#10,#10,#60 ; y
	db #00,#00,#7c,#08,#10,#20,#7c,#00 ; z
	db #0c,#10,#10,#60,#10,#10,#0c,#00 ; {
	db #10,#10,#10,#10,#10,#10,#10,#00 ; |
	db #60,#10,#10,#0c,#10,#10,#60,#00 ; }
	db #00,#00,#24,#54,#48,#00,#00,#00 ; ~
	db #3c,#5a,#a5,#a1,#a5,#99,#42,#3c ; ©   No copyright sign at 7F
	;db #00,#00,#00,#00,#00,#00,#00,#00 ; for 7F if no copyright sign used
	
font_end:
 
 ;*** ZX Spectrum application will follow here ***
 